## Repo overview

- Framework: Laravel (composer.json requires "laravel/framework": "^11.9"). PHP 8.2+ is required.
- Project root for the web app lives in `site_files/` (this repo contains a packaged Laravel app under `site_files`). The web entrypoint is `site_files/public`.
- Key areas: `site_files/app/` (app code / controllers), `site_files/resources/views/` (Blade templates), `site_files/routes/web.php` (routing), `site_files/app/Helpers/` (project-specific helper functions), `site_files/config/` (app config), and `site_files/storage/` (uploads).

## What to know to be productive

- This is a monolithic Laravel app with many custom helpers. Many Blade templates call project helpers rather than Laravel facades. Search `site_files/app/Helpers` for the implementation before editing view logic.
  - Examples in `site_files/resources/views/front/home/index.blade.php`: `asset_uploads('module/banner/' . $banner->featured_img)`, `adjustUrl($banner->content)`, and `storage_uploads('blog/' . $blogsValues['featured_img'])`.
  - Common helpers are in `site_files/app/Helpers/` (e.g. `mysite_helper.php`, `blog_helper.php`, `image_uploader_helper.php`, `DashboardLinks.php`).

- Routing and controllers: `site_files/routes/web.php` contains the monolithic route definitions (admin routes under `adminmedia` prefix). When you add or refactor routes, update this file and search for matching controller namespaces under `site_files/app/Http/Controllers/Back` and `site_files/app/Http/Controllers/Front`.

- Package/dependency constraints: see `site_files/composer.json`. Notable packages: Laravel 11, Intervention/Image, Yajra Datatables, maatwebsite/excel, Stripe, Vonage, Spatie packages. Prefer compatibility with PHP 8.2 and Laravel 11 when adding packages or refactoring.

## Local dev / build hints

- Typical Laravel commands apply (run from `site_files/`):
  - `composer install`
  - copy `.env.example` to `.env` and set DB/APP URL values
  - `php artisan key:generate`
  - `php artisan migrate --force` (or without `--force` locally)
  - `php artisan storage:link` if using storage symlinks

- This project is often run under XAMPP/Apache with DocumentRoot pointing to `site_files/public` (your workspace root contains `/Applications/XAMPP/xamppfiles/htdocs/catalyst` — ensure Apache is configured accordingly or use `php artisan serve` for quick local testing).

## Conventions and patterns in this codebase

- Helpers-first view logic: Views frequently call global helpers for assets, URL adjustments and excerpts rather than Eloquent accessors. Example patterns to respect:
  - asset and storage helpers: `asset_uploads(...)`, `storage_uploads(...)` — don't replace these blindly with `asset()` unless you audit their behaviour.
  - HTML/content sanitization is handled by helpers like `adjustUrl()` and the `masterro/laravel-xss-filter` package is installed; prefer those helpers.

- Blade: templates are under `site_files/resources/views/front` and `site_files/resources/views/back`. Keep markup and small display logic in Blade; move data-heavy logic into controllers or helpers.

- Files uploaded by admin are stored under `site_files/storage/app` with helpers that build URLs. Search for `asset_uploads(` and `storage_uploads(` to find how uploads are referenced.

## Tests & linting

- PHPUnit tests are present under `site_files/tests/` and composer.json requires `phpunit/phpunit`. Run tests from `site_files/`: `./vendor/bin/phpunit`.
- The repo includes `laravel/pint` in require-dev; use `./vendor/bin/pint` for formatting checks compatible with the project.

## When you edit code, keep these checks in mind

- Preserve helper contracts: changes to helpers in `site_files/app/Helpers` may affect many views. Run a quick grep for usages before renaming.
- Because routes are centralized in `site_files/routes/web.php`, adding controllers should include appropriate route declarations and controller namespace imports consistent with the file's style (look at existing grouped admin routes using `adminmedia` prefix).

## Quick references (examples)

- Banner image in Blade: `{{ asset_uploads('module/banner/' . $banner->featured_img) }}` — helper builds public URL for admin-uploaded banner images.
- Blog image existence check: `file_exists(storage_uploads('blog/' . $blogsValues['featured_img']))` then `asset_uploads('blog/' . $blogsValues['featured_img'])` for final src.

## Where to look for more context

- `site_files/composer.json` — project dependencies and scripts
- `site_files/routes/web.php` — full route map and controller organization
- `site_files/app/Helpers/` — canonical place for cross-cutting helpers
- `site_files/resources/views/front/` — common front-end view patterns (see `index.blade.php` for many examples)

---
If you'd like, I can (1) merge this into an existing instructions file if you have one elsewhere, (2) add a short troubleshooting section (DB, .env, XAMPP-specific notes), or (3) extract a short checklist for PR reviewers (what tests to run and what to watch for). Which would you prefer? 
